function [b varargout] = betweennessCentrality(obj, varargin)
%BETWEENNESSCENTRALITY Calculate betweenness centrality for the
%AdjacencyMatrix object obj
%   b = BETWEENNESSCENTRALITY(obj,neuronName) will calculate the
%       betweenness centrality of the neuron given by neuronName in
%       AdjacencyMatrix obj.  This function calls the function
%       betweenness_wei, part of the brain connectivity toolbox.  Weights
%       in the adjacency matrix are considered to be the inverse of
%       distance in calculating shortest paths (ie. high weight means low
%       distance)
%
%   [b, labels] = BETWEENNESSCENTRALITY(obj,n) will return the n neurons
%   	with highest betweenness centrality.  b will be a nx1 vector of
%   	betweenness centrality values, and labels will be a nx1 cell array
%   	giving their names
%
%       BETWEENNESSCENTRALITY(...,'quiet') will suppress output


% If necessary, calculate betweenness centrality for AdjacencyMatrix obj

if isempty(obj.BC)
    
    [A,obj.BClabels] = getSquareMatrix(obj);
    I = find(A);
    % redefine weight matrix as distance matrix (distance = 1/weight)
    A(I) = 1./A(I);
    obj.BC = betweenness_wei(A);
    % sort betweenness centrality data
    [obj.BC idx] = sort(obj.BC,'descend');
    obj.BClabels = obj.BClabels(idx);
end

if nargin == 1
    b = obj.BC;
    varargout{1} = obj.BClabels;
    for i = 1:length(obj.BC)
        fprintf('%-12.4f%s\n',obj.BC(i),obj.BClabels{i})
    end
    
    return
end

switch class(varargin{1})
    case 'char'
        if ~any(strcmp(obj.BClabels,varargin{1}))
            error('AdjacencyMatrix:NeuronLookup',['Could not find neuron ' varargin{1}]);
        end
        
        b = obj.BC(strcmp(obj.BClabels,varargin{1}));
        return
    case 'double'
        b = obj.BC(1:varargin{1});
        varargout{1} = obj.BClabels(1:varargin{1});
        for i = 1:length(b)
            fprintf('%-12.4f%s\n',b(i),varargout{1}{i})
        end
end

end

